DROP PROCEDURE [hips].[IhiDuplicatePatientCount]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Simon Biber
-- Create date:   23 October 2012
-- Description:   Returns the number of possible 
--				  Duplicated patients in the system.
--
-- This is used to implement HI CR 5875:
--     If an IHI with a supported record status is returned from the HI Service
--     for a patient, the software shall have the capacity to assign that IHI
--     to the patient's record and raise an alert if the search criteria used
--     matches another patient?s demographic data from the same registration
--     source. If an alert is raised, the system shall either discard the IHI
--     or store it against the target patient record and flag the records as
--     potentially conflicting.
--
-- HIPS can treat each hospital as a separate registration source, in which
-- case this stored procedure will only count duplicates that are registered
-- in the same hospital as an MRN on the patient master whose IHI is being
-- searched for.
-- 
-- Modified:
-- 02/04/2013 - sbiber - If this IHI was provided by another system
--                       it will have no MedicareNumber or DvaNumber
--                       however this should not be considered a
--                       potential duplicate patient record.
-- 27/05/2013 - sbiber - Add ability to filter by hospital.
-- 26/06/2013 - sbiber - Change the filter by hospital to a filter that counts
--                       those PMs which have MRNs from the same hospital as
--                       any MRN on the PM which is being searched.
-- =============================================
CREATE PROCEDURE [hips].[IhiDuplicatePatientCount] 
(
	@PatientMasterId INT,
	@LastName VARCHAR(80),
	@FirstNames VARCHAR(80),
	@MedicareNumber VARCHAR(12) = NULL,
	@MedicareIrn VARCHAR(3) = NULL,
	@DvaNumber VARCHAR(18) = NULL,
	@DateOfBirth DATETIME
)
AS
BEGIN
	IF @MedicareNumber IS NULL AND @DvaNumber IS NULL
	BEGIN
		SELECT 0 AS DuplicatePatients
	END
	ELSE
	BEGIN
		SELECT COUNT(DISTINCT dupPM.PatientMasterId) AS DuplicatePatients 
		FROM hips.PatientMaster dupPM WITH (NOLOCK)
		INNER JOIN hips.PatientMasterName pmn  WITH (NOLOCK) ON dupPM.PatientMasterId = pmn.PatientMasterId
		AND pmn.NameTypeId = 2  -- Current Name in PAS
		INNER JOIN hips.HospitalPatient dupHP on dupPM.PatientMasterId = dupHP.PatientMasterId
		INNER JOIN hips.HospitalPatient thisHP on thisHP.HospitalId = dupHP.HospitalId -- Has an HP from same hospital as this PM
		
		WHERE @LastName = pmn.FamilyName
		AND pmn.GivenNames LIKE '%' + RTRIM(@FirstNames) + '%'
		AND (@MedicareNumber = dupPM.MedicareNumber OR @MedicareNumber IS NULL)
		AND (@MedicareIrn = dupPM.MedicareIrn OR @MedicareIrn IS NULL)
		AND (@DvaNumber = dupPM.DvaNumber OR @DvaNumber IS NULL)
		AND @DateOfBirth = dupPM.DateOfBirth
		AND thisHP.PatientMasterId = @PatientMasterId  -- This PM is the one whose IHI is being searched for.
		AND dupPM.PatientMasterId <> @PatientMasterId  -- Don't count a duplicate with itself.
	END
END
GO

